<?php
/**
 * صفحه مدیریت کانفیگ‌ها
 */
function renderAdminConfigs() {
?>
<main class="pr-0 lg:pr-64 pt-16 min-h-screen">
    <div class="p-4 lg:p-8">
        <!-- هدر صفحه -->
        <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-4 mb-8">
            <div>
                <h1 class="text-2xl font-bold">مدیریت کانفیگ‌ها</h1>
                <p class="text-gray-400 mt-1">افزودن و مدیریت کانفیگ‌های VPN</p>
            </div>
            <div class="flex items-center gap-3">
                <button onclick="openAddConfigModal()" class="bg-gradient-to-r from-primary-500 to-secondary-500 rounded-xl px-6 py-2.5 font-medium hover:opacity-90 transition-opacity flex items-center gap-2">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                    </svg>
                    افزودن کانفیگ
                </button>
                <button onclick="openBulkAddModal()" class="glass rounded-xl px-4 py-2.5 hover:bg-white/10 transition-all flex items-center gap-2">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-8l-4-4m0 0L8 8m4-4v12"></path>
                    </svg>
                    افزودن دسته‌ای
                </button>
            </div>
        </div>

        <!-- آمار کانفیگ‌ها -->
        <div class="grid grid-cols-2 md:grid-cols-5 gap-4 mb-6">
            <div class="glass rounded-2xl p-4 text-center">
                <div class="text-3xl font-bold text-primary-400" id="totalConfigs">0</div>
                <div class="text-sm text-gray-400">کل کانفیگ‌ها</div>
            </div>
            <div class="glass rounded-2xl p-4 text-center">
                <div class="text-3xl font-bold text-green-400" id="availableConfigs">0</div>
                <div class="text-sm text-gray-400">موجود</div>
            </div>
            <div class="glass rounded-2xl p-4 text-center">
                <div class="text-3xl font-bold text-blue-400" id="soldConfigs">0</div>
                <div class="text-sm text-gray-400">فروخته شده</div>
            </div>
            <div class="glass rounded-2xl p-4 text-center">
                <div class="text-3xl font-bold text-yellow-400" id="reservedConfigs">0</div>
                <div class="text-sm text-gray-400">رزرو شده</div>
            </div>
            <div class="glass rounded-2xl p-4 text-center">
                <div class="text-3xl font-bold text-red-400" id="expiredConfigs">0</div>
                <div class="text-sm text-gray-400">منقضی</div>
            </div>
        </div>

        <!-- تب‌های نوع کانفیگ -->
        <div class="flex items-center gap-2 mb-6 overflow-x-auto pb-2">
            <button onclick="filterByType('all')" class="config-type-tab active px-6 py-2.5 rounded-xl bg-primary-500 text-white font-medium whitespace-nowrap" data-type="all">
                همه
            </button>
            <button onclick="filterByType('vless')" class="config-type-tab px-6 py-2.5 rounded-xl glass hover:bg-white/10 font-medium whitespace-nowrap" data-type="vless">
                <span class="flex items-center gap-2">
                    <span class="w-2 h-2 rounded-full bg-blue-400"></span>
                    VLESS
                </span>
            </button>
            <button onclick="filterByType('vmess')" class="config-type-tab px-6 py-2.5 rounded-xl glass hover:bg-white/10 font-medium whitespace-nowrap" data-type="vmess">
                <span class="flex items-center gap-2">
                    <span class="w-2 h-2 rounded-full bg-green-400"></span>
                    VMess
                </span>
            </button>
            <button onclick="filterByType('trojan')" class="config-type-tab px-6 py-2.5 rounded-xl glass hover:bg-white/10 font-medium whitespace-nowrap" data-type="trojan">
                <span class="flex items-center gap-2">
                    <span class="w-2 h-2 rounded-full bg-purple-400"></span>
                    Trojan
                </span>
            </button>
            <button onclick="filterByType('shadowsocks')" class="config-type-tab px-6 py-2.5 rounded-xl glass hover:bg-white/10 font-medium whitespace-nowrap" data-type="shadowsocks">
                <span class="flex items-center gap-2">
                    <span class="w-2 h-2 rounded-full bg-yellow-400"></span>
                    Shadowsocks
                </span>
            </button>
            <button onclick="filterByType('netmod')" class="config-type-tab px-6 py-2.5 rounded-xl glass hover:bg-white/10 font-medium whitespace-nowrap" data-type="netmod">
                <span class="flex items-center gap-2">
                    <span class="w-2 h-2 rounded-full bg-pink-400"></span>
                    NetMod
                </span>
            </button>
        </div>

        <!-- فیلترها -->
        <div class="glass rounded-3xl p-6 mb-6">
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4">
                <div class="relative">
                    <input type="text" id="searchConfigs" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 pr-12 focus:border-primary-500 outline-none" placeholder="جستجو...">
                    <svg class="w-5 h-5 absolute right-4 top-1/2 -translate-y-1/2 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
                    </svg>
                </div>
                <select id="configStatusFilter" class="bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none">
                    <option value="">همه وضعیت‌ها</option>
                    <option value="available">موجود</option>
                    <option value="sold">فروخته شده</option>
                    <option value="reserved">رزرو شده</option>
                    <option value="expired">منقضی</option>
                </select>
                <select id="configLocationFilter" class="bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none">
                    <option value="">همه لوکیشن‌ها</option>
                    <!-- لوکیشن‌ها با AJAX لود می‌شوند -->
                </select>
                <select id="configSortFilter" class="bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none">
                    <option value="newest">جدیدترین</option>
                    <option value="oldest">قدیمی‌ترین</option>
                    <option value="price_asc">ارزان‌ترین</option>
                    <option value="price_desc">گران‌ترین</option>
                    <option value="most_sold">پرفروش‌ترین</option>
                </select>
            </div>
        </div>

        <!-- جدول کانفیگ‌ها -->
        <div class="glass rounded-3xl overflow-hidden">
            <div class="overflow-x-auto">
                <table class="w-full">
                    <thead>
                        <tr class="text-gray-400 text-sm border-b border-white/10">
                            <th class="text-right p-4">
                                <input type="checkbox" id="selectAllConfigs" class="rounded bg-white/10 border-white/20">
                            </th>
                            <th class="text-right p-4">نام / شناسه</th>
                            <th class="text-right p-4">نوع</th>
                            <th class="text-right p-4">لوکیشن</th>
                            <th class="text-right p-4">قیمت</th>
                            <th class="text-right p-4">موجودی</th>
                            <th class="text-right p-4">فروش</th>
                            <th class="text-right p-4">وضعیت</th>
                            <th class="text-center p-4">عملیات</th>
                        </tr>
                    </thead>
                    <tbody id="configsTable">
                        <!-- داده‌ها با AJAX لود می‌شوند -->
                    </tbody>
                </table>
            </div>
            
            <!-- صفحه‌بندی -->
            <div class="p-4 border-t border-white/10">
                <div class="flex flex-col md:flex-row items-center justify-between gap-4">
                    <span class="text-gray-400 text-sm" id="configsCount">نمایش 0 کانفیگ</span>
                    <div class="flex items-center gap-2" id="configsPagination"></div>
                </div>
            </div>
        </div>
    </div>
</main>

<!-- Modal افزودن/ویرایش کانفیگ -->
<div id="configModal" class="fixed inset-0 z-50 hidden items-center justify-center p-4 bg-black/50 backdrop-blur-sm">
    <div class="glass rounded-3xl p-6 w-full max-w-2xl animate-scale-in max-h-[90vh] overflow-y-auto">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-lg font-bold" id="configModalTitle">افزودن کانفیگ جدید</h3>
            <button onclick="closeConfigModal()" class="w-8 h-8 rounded-full bg-white/10 flex items-center justify-center hover:bg-red-500/20">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </button>
        </div>
        <form id="configForm" class="space-y-4">
            <input type="hidden" name="config_id" id="editConfigId">
            
            <div class="grid grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm text-gray-400 mb-2">نام کانفیگ <span class="text-red-400">*</span></label>
                    <input type="text" name="name" id="configName" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" required>
                </div>
                <div>
                    <label class="block text-sm text-gray-400 mb-2">نوع پروتکل <span class="text-red-400">*</span></label>
                    <select name="type" id="configType" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" required>
                        <option value="vless">VLESS</option>
                        <option value="vmess">VMess</option>
                        <option value="trojan">Trojan</option>
                        <option value="shadowsocks">Shadowsocks</option>
                        <option value="netmod">NetMod</option>
                    </select>
                </div>
            </div>
            
            <div class="grid grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm text-gray-400 mb-2">لوکیشن <span class="text-red-400">*</span></label>
                    <select name="location" id="configLocation" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" required>
                        <option value="germany">🇩🇪 آلمان</option>
                        <option value="netherlands">🇳🇱 هلند</option>
                        <option value="france">🇫🇷 فرانسه</option>
                        <option value="usa">🇺🇸 آمریکا</option>
                        <option value="uk">🇬🇧 انگلستان</option>
                        <option value="canada">🇨🇦 کانادا</option>
                        <option value="singapore">🇸🇬 سنگاپور</option>
                        <option value="turkey">🇹🇷 ترکیه</option>
                        <option value="uae">🇦🇪 امارات</option>
                    </select>
                </div>
                <div>
                    <label class="block text-sm text-gray-400 mb-2">قیمت (تومان) <span class="text-red-400">*</span></label>
                    <input type="number" name="price" id="configPrice" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" required min="0">
                </div>
            </div>
            
            <div class="grid grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm text-gray-400 mb-2">مدت زمان (روز) <span class="text-red-400">*</span></label>
                    <input type="number" name="duration" id="configDuration" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" value="30" required min="1">
                </div>
                <div>
                    <label class="block text-sm text-gray-400 mb-2">حجم ترافیک (GB)</label>
                    <input type="number" name="traffic" id="configTraffic" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" placeholder="خالی = نامحدود">
                </div>
            </div>
            
            <div class="grid grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm text-gray-400 mb-2">تعداد موجودی</label>
                    <input type="number" name="stock" id="configStock" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" value="1" min="1">
                    <p class="text-xs text-gray-500 mt-1">چند بار این کانفیگ قابل فروش است</p>
                </div>
                <div>
                    <label class="block text-sm text-gray-400 mb-2">حداکثر دستگاه</label>
                    <input type="number" name="max_devices" id="configMaxDevices" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" value="2" min="1">
                </div>
            </div>
            
            <div>
                <label class="block text-sm text-gray-400 mb-2">لینک کانفیگ <span class="text-red-400">*</span></label>
                <textarea name="config_data" id="configData" rows="4" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none resize-none font-mono text-sm" required placeholder="vless://... یا vmess://..."></textarea>
            </div>
            
            <div>
                <label class="block text-sm text-gray-400 mb-2">توضیحات</label>
                <textarea name="description" id="configDescription" rows="2" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none resize-none"></textarea>
            </div>
            
            <div class="flex items-center gap-4">
                <label class="flex items-center gap-2 cursor-pointer">
                    <input type="checkbox" name="is_active" id="configIsActive" class="rounded bg-white/10 border-white/20" checked>
                    <span class="text-sm">فعال</span>
                </label>
                <label class="flex items-center gap-2 cursor-pointer">
                    <input type="checkbox" name="is_featured" id="configIsFeatured" class="rounded bg-white/10 border-white/20">
                    <span class="text-sm">ویژه</span>
                </label>
            </div>
            
            <div class="flex gap-3 pt-4">
                <button type="submit" class="flex-1 bg-gradient-to-r from-primary-500 to-secondary-500 rounded-xl py-3 font-medium hover:opacity-90 transition-opacity">
                    ذخیره کانفیگ
                </button>
                <button type="button" onclick="closeConfigModal()" class="flex-1 glass rounded-xl py-3 font-medium hover:bg-white/10">
                    انصراف
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Modal افزودن دسته‌ای -->
<div id="bulkAddModal" class="fixed inset-0 z-50 hidden items-center justify-center p-4 bg-black/50 backdrop-blur-sm">
    <div class="glass rounded-3xl p-6 w-full max-w-2xl animate-scale-in max-h-[90vh] overflow-y-auto">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-lg font-bold">افزودن دسته‌ای کانفیگ</h3>
            <button onclick="closeBulkAddModal()" class="w-8 h-8 rounded-full bg-white/10 flex items-center justify-center hover:bg-red-500/20">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </button>
        </div>
        <form id="bulkAddForm" class="space-y-4">
            <div class="grid grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm text-gray-400 mb-2">نام پایه</label>
                    <input type="text" name="base_name" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" placeholder="مثال: سرور آلمان" required>
                </div>
                <div>
                    <label class="block text-sm text-gray-400 mb-2">نوع پروتکل</label>
                    <select name="type" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" required>
                        <option value="vless">VLESS</option>
                        <option value="vmess">VMess</option>
                        <option value="trojan">Trojan</option>
                    </select>
                </div>
            </div>
            
            <div class="grid grid-cols-3 gap-4">
                <div>
                    <label class="block text-sm text-gray-400 mb-2">لوکیشن</label>
                    <select name="location" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" required>
                        <option value="germany">🇩🇪 آلمان</option>
                        <option value="netherlands">🇳🇱 هلند</option>
                        <option value="france">🇫🇷 فرانسه</option>
                        <option value="usa">🇺🇸 آمریکا</option>
                    </select>
                </div>
                <div>
                    <label class="block text-sm text-gray-400 mb-2">قیمت (تومان)</label>
                    <input type="number" name="price" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" required>
                </div>
                <div>
                    <label class="block text-sm text-gray-400 mb-2">مدت (روز)</label>
                    <input type="number" name="duration" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none" value="30" required>
                </div>
            </div>
            
            <div>
                <label class="block text-sm text-gray-400 mb-2">کانفیگ‌ها (هر خط یک کانفیگ)</label>
                <textarea name="configs" rows="10" class="w-full bg-white/5 border border-white/10 rounded-xl px-4 py-3 focus:border-primary-500 outline-none resize-none font-mono text-sm" placeholder="vless://...
vmess://...
trojan://..." required></textarea>
                <p class="text-xs text-gray-500 mt-1">هر کانفیگ را در یک خط جداگانه وارد کنید</p>
            </div>
            
            <div class="flex gap-3 pt-4">
                <button type="submit" class="flex-1 bg-gradient-to-r from-primary-500 to-secondary-500 rounded-xl py-3 font-medium hover:opacity-90 transition-opacity">
                    افزودن همه
                </button>
                <button type="button" onclick="closeBulkAddModal()" class="flex-1 glass rounded-xl py-3 font-medium hover:bg-white/10">
                    انصراف
                </button>
            </div>
        </form>
    </div>
</div>

<script>
let currentConfigPage = 1;
let currentConfigType = 'all';
const configsPerPage = 20;

// لود کانفیگ‌ها
async function loadConfigs(page = 1) {
    currentConfigPage = page;
    const status = document.getElementById('configStatusFilter').value;
    const location = document.getElementById('configLocationFilter').value;
    const sort = document.getElementById('configSortFilter').value;
    const search = document.getElementById('searchConfigs').value;
    
    try {
        const response = await fetch(`api.php?action=admin_get_configs&page=${page}&limit=${configsPerPage}&type=${currentConfigType}&status=${status}&location=${location}&sort=${sort}&search=${encodeURIComponent(search)}`);
        const data = await response.json();
        
        if (data.success) {
            renderConfigsTable(data.configs);
            renderConfigsPagination(data.total, page);
            document.getElementById('configsCount').textContent = `نمایش ${data.configs.length} از ${data.total} کانفیگ`;
            
            // آمار
            document.getElementById('totalConfigs').textContent = data.stats.total;
            document.getElementById('availableConfigs').textContent = data.stats.available;
            document.getElementById('soldConfigs').textContent = data.stats.sold;
            document.getElementById('reservedConfigs').textContent = data.stats.reserved;
            document.getElementById('expiredConfigs').textContent = data.stats.expired;
        }
    } catch (error) {
        console.error('Error loading configs:', error);
    }
}

// فیلتر بر اساس نوع
function filterByType(type) {
    currentConfigType = type;
    document.querySelectorAll('.config-type-tab').forEach(tab => {
        if (tab.dataset.type === type) {
            tab.classList.add('active', 'bg-primary-500', 'text-white');
            tab.classList.remove('glass');
        } else {
            tab.classList.remove('active', 'bg-primary-500', 'text-white');
            tab.classList.add('glass');
        }
    });
    loadConfigs(1);
}

function renderConfigsTable(configs) {
    const tbody = document.getElementById('configsTable');
    
    if (configs.length === 0) {
        tbody.innerHTML = `
            <tr>
                <td colspan="9" class="text-center py-12 text-gray-400">
                    <svg class="w-16 h-16 mx-auto mb-4 opacity-50" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                    </svg>
                    کانفیگی یافت نشد
                </td>
            </tr>
        `;
        return;
    }
    
    tbody.innerHTML = configs.map(config => `
        <tr class="border-b border-white/5 hover:bg-white/5">
            <td class="p-4">
                <input type="checkbox" class="config-checkbox rounded bg-white/10 border-white/20" data-id="${config.id}">
            </td>
            <td class="p-4">
                <div>
                    <div class="font-medium flex items-center gap-2">
                        ${config.name}
                        ${config.is_featured ? '<span class="px-1.5 py-0.5 rounded text-xs bg-yellow-500/20 text-yellow-400">ویژه</span>' : ''}
                    </div>
                    <div class="text-xs text-gray-500 font-mono">#${config.id}</div>
                </div>
            </td>
            <td class="p-4">
                <span class="px-2 py-1 rounded-lg text-xs ${getConfigTypeClass(config.type)}">
                    ${config.type.toUpperCase()}
                </span>
            </td>
            <td class="p-4">
                <span class="flex items-center gap-2">
                    <span>${getLocationFlag(config.location)}</span>
                    <span>${getLocationName(config.location)}</span>
                </span>
            </td>
            <td class="p-4">
                <span class="font-medium">${numberFormat(config.price)}</span>
                <span class="text-xs text-gray-400">تومان</span>
            </td>
            <td class="p-4">
                <span class="font-medium ${config.stock > 0 ? 'text-green-400' : 'text-red-400'}">${config.stock}</span>
            </td>
            <td class="p-4">
                <span class="text-blue-400">${config.sold_count}</span>
            </td>
            <td class="p-4">
                <span class="px-3 py-1 rounded-full text-xs ${getConfigStatusClass(config.status)}">
                    ${getConfigStatusText(config.status)}
                </span>
            </td>
            <td class="p-4">
                <div class="flex items-center justify-center gap-1">
                    <button onclick="copyConfig(${config.id})" class="w-8 h-8 rounded-lg bg-gray-500/20 text-gray-400 flex items-center justify-center hover:bg-gray-500/30" title="کپی">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 5H6a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2v-1M8 5a2 2 0 002 2h2a2 2 0 002-2M8 5a2 2 0 012-2h2a2 2 0 012 2m0 0h2a2 2 0 012 2v3m2 4H10m0 0l3-3m-3 3l3 3"></path>
                        </svg>
                    </button>
                    <button onclick="editConfig(${config.id})" class="w-8 h-8 rounded-lg bg-yellow-500/20 text-yellow-400 flex items-center justify-center hover:bg-yellow-500/30" title="ویرایش">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"></path>
                        </svg>
                    </button>
                    <button onclick="toggleConfigStatus(${config.id})" class="w-8 h-8 rounded-lg ${config.is_active ? 'bg-red-500/20 text-red-400' : 'bg-green-500/20 text-green-400'} flex items-center justify-center hover:opacity-80" title="${config.is_active ? 'غیرفعال' : 'فعال'}">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            ${config.is_active ? 
                                '<path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M18.364 18.364A9 9 0 005.636 5.636m12.728 12.728A9 9 0 015.636 5.636m12.728 12.728L5.636 5.636"></path>' :
                                '<path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>'
                            }
                        </svg>
                    </button>
                    <button onclick="deleteConfig(${config.id})" class="w-8 h-8 rounded-lg bg-red-500/20 text-red-400 flex items-center justify-center hover:bg-red-500/30" title="حذف">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                        </svg>
                    </button>
                </div>
            </td>
        </tr>
    `).join('');
}

// Modal افزودن کانفیگ
function openAddConfigModal() {
    document.getElementById('configModalTitle').textContent = 'افزودن کانفیگ جدید';
    document.getElementById('configForm').reset();
    document.getElementById('editConfigId').value = '';
    document.getElementById('configIsActive').checked = true;
    document.getElementById('configModal').classList.remove('hidden');
    document.getElementById('configModal').classList.add('flex');
}

function closeConfigModal() {
    document.getElementById('configModal').classList.add('hidden');
    document.getElementById('configModal').classList.remove('flex');
}

// ویرایش کانفیگ
async function editConfig(configId) {
    try {
        const response = await fetch(`api.php?action=admin_get_config&id=${configId}`);
        const data = await response.json();
        
        if (data.success) {
            const config = data.config;
            document.getElementById('configModalTitle').textContent = 'ویرایش کانفیگ';
            document.getElementById('editConfigId').value = config.id;
            document.getElementById('configName').value = config.name;
            document.getElementById('configType').value = config.type;
            document.getElementById('configLocation').value = config.location;
            document.getElementById('configPrice').value = config.price;
            document.getElementById('configDuration').value = config.duration;
            document.getElementById('configTraffic').value = config.traffic || '';
            document.getElementById('configStock').value = config.stock;
            document.getElementById('configMaxDevices').value = config.max_devices;
            document.getElementById('configData').value = config.config_data;
            document.getElementById('configDescription').value = config.description || '';
            document.getElementById('configIsActive').checked = config.is_active;
            document.getElementById('configIsFeatured').checked = config.is_featured;
            
            document.getElementById('configModal').classList.remove('hidden');
            document.getElementById('configModal').classList.add('flex');
        }
    } catch (error) {
        showToast('خطا در دریافت اطلاعات', 'error');
    }
}

// ذخیره کانفیگ
document.getElementById('configForm')?.addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    formData.append('action', formData.get('config_id') ? 'admin_update_config' : 'admin_add_config');
    formData.append('is_active', document.getElementById('configIsActive').checked ? '1' : '0');
    formData.append('is_featured', document.getElementById('configIsFeatured').checked ? '1' : '0');
    
    try {
        const response = await fetch('api.php', { method: 'POST', body: formData });
        const data = await response.json();
        
        if (data.success) {
            showToast(data.message, 'success');
            closeConfigModal();
            loadConfigs(currentConfigPage);
        } else {
            showToast(data.message, 'error');
        }
    } catch (error) {
        showToast('خطا در ذخیره اطلاعات', 'error');
    }
});

// Modal افزودن دسته‌ای
function openBulkAddModal() {
    document.getElementById('bulkAddForm').reset();
    document.getElementById('bulkAddModal').classList.remove('hidden');
    document.getElementById('bulkAddModal').classList.add('flex');
}

function closeBulkAddModal() {
    document.getElementById('bulkAddModal').classList.add('hidden');
    document.getElementById('bulkAddModal').classList.remove('flex');
}

// ذخیره دسته‌ای
document.getElementById('bulkAddForm')?.addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    formData.append('action', 'admin_bulk_add_configs');
    
    try {
        const response = await fetch('api.php', { method: 'POST', body: formData });
        const data = await response.json();
        
        if (data.success) {
            showToast(`${data.count} کانفیگ با موفقیت اضافه شد`, 'success');
            closeBulkAddModal();
            loadConfigs(1);
        } else {
            showToast(data.message, 'error');
        }
    } catch (error) {
        showToast('خطا در افزودن دسته‌ای', 'error');
    }
});

// حذف کانفیگ
async function deleteConfig(configId) {
    if (!confirm('آیا از حذف این کانفیگ اطمینان دارید؟')) return;
    
    try {
        const response = await fetch('api.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ action: 'admin_delete_config', config_id: configId })
        });
        const data = await response.json();
        
        if (data.success) {
            showToast('کانفیگ حذف شد', 'success');
            loadConfigs(currentConfigPage);
        } else {
            showToast(data.message, 'error');
        }
    } catch (error) {
        showToast('خطا در حذف کانفیگ', 'error');
    }
}

// کپی کانفیگ
async function copyConfig(configId) {
    try {
        const response = await fetch(`api.php?action=admin_get_config&id=${configId}`);
        const data = await response.json();
        
        if (data.success) {
            await navigator.clipboard.writeText(data.config.config_data);
            showToast('کانفیگ کپی شد', 'success');
        }
    } catch (error) {
        showToast('خطا در کپی کانفیگ', 'error');
    }
}

// توابع کمکی
function getConfigTypeClass(type) {
    const classes = {
        'vless': 'bg-blue-500/20 text-blue-400',
        'vmess': 'bg-green-500/20 text-green-400',
        'trojan': 'bg-purple-500/20 text-purple-400',
        'shadowsocks': 'bg-yellow-500/20 text-yellow-400',
        'netmod': 'bg-pink-500/20 text-pink-400'
    };
    return classes[type] || classes['vless'];
}

function getConfigStatusClass(status) {
    const classes = {
        'available': 'bg-green-500/20 text-green-400',
        'sold': 'bg-blue-500/20 text-blue-400',
        'reserved': 'bg-yellow-500/20 text-yellow-400',
        'expired': 'bg-red-500/20 text-red-400'
    };
    return classes[status] || classes['available'];
}

function getConfigStatusText(status) {
    const texts = { 'available': 'موجود', 'sold': 'فروخته شده', 'reserved': 'رزرو', 'expired': 'منقضی' };
    return texts[status] || status;
}

function getLocationFlag(location) {
    const flags = {
        'germany': '🇩🇪', 'netherlands': '🇳🇱', 'france': '🇫🇷', 'usa': '🇺🇸',
        'uk': '🇬🇧', 'canada': '🇨🇦', 'singapore': '🇸🇬', 'turkey': '🇹🇷', 'uae': '🇦🇪'
    };
    return flags[location] || '🌍';
}

function getLocationName(location) {
    const names = {
        'germany': 'آلمان', 'netherlands': 'هلند', 'france': 'فرانسه', 'usa': 'آمریکا',
        'uk': 'انگلستان', 'canada': 'کانادا', 'singapore': 'سنگاپور', 'turkey': 'ترکیه', 'uae': 'امارات'
    };
    return names[location] || location;
}

// Event Listeners
document.getElementById('configStatusFilter')?.addEventListener('change', () => loadConfigs(1));
document.getElementById('configLocationFilter')?.addEventListener('change', () => loadConfigs(1));
document.getElementById('configSortFilter')?.addEventListener('change', () => loadConfigs(1));
document.getElementById('searchConfigs')?.addEventListener('input', debounce(() => loadConfigs(1), 500));

// لود اولیه
document.addEventListener('DOMContentLoaded', () => loadConfigs());
</script>
<?php
}
?>