<?php
/**
 * API Backend
 * فایل پردازش درخواست‌های AJAX
 */

session_start();
header('Content-Type: application/json; charset=utf-8');

require_once 'config.php';
require_once 'functions.php';

// دریافت action
$action = $_GET['action'] ?? $_POST['action'] ?? '';

// بررسی احراز هویت
$publicActions = ['login', 'register', 'get_products', 'check_discount', 'get_bank_card'];
$user = null;

if (!in_array($action, $publicActions)) {
    if (!isset($_SESSION['user_id'])) {
        jsonResponse(false, 'لطفا وارد شوید');
    }
    $user = getUserById($_SESSION['user_id']);
    if (!$user) {
        jsonResponse(false, 'کاربر یافت نشد');
    }
}

// پردازش درخواست‌ها
switch ($action) {
    
    // ============ احراز هویت ============
    case 'login':
        $username = sanitize($_POST['username'] ?? '');
        $password = $_POST['password'] ?? '';
        
        if (empty($username) || empty($password)) {
            jsonResponse(false, 'لطفا تمام فیلدها را پر کنید');
        }
        
        $user = getUserByUsername($username);
        if (!$user || !password_verify($password, $user['password'])) {
            jsonResponse(false, 'نام کاربری یا رمز عبور اشتباه است');
        }
        
        if ($user['status'] === 'blocked') {
            jsonResponse(false, 'حساب کاربری شما مسدود شده است');
        }
        
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['is_admin'] = $user['role'] === 'admin';
        
        updateLastLogin($user['id']);
        
        jsonResponse(true, 'ورود موفق', ['redirect' => $user['role'] === 'admin' ? '?page=admin' : '?page=dashboard']);
        break;
        
    case 'register':
        $username = sanitize($_POST['username'] ?? '');
        $phone = sanitize($_POST['phone'] ?? '');
        $password = $_POST['password'] ?? '';
        $referral_code = sanitize($_POST['referral_code'] ?? '');
        
        if (empty($username) || empty($phone) || empty($password)) {
            jsonResponse(false, 'لطفا تمام فیلدها را پر کنید');
        }
        
        if (strlen($password) < 6) {
            jsonResponse(false, 'رمز عبور باید حداقل 6 کاراکتر باشد');
        }
        
        if (getUserByUsername($username)) {
            jsonResponse(false, 'این نام کاربری قبلا ثبت شده است');
        }
        
        if (getUserByPhone($phone)) {
            jsonResponse(false, 'این شماره موبایل قبلا ثبت شده است');
        }
        
        $referrer_id = null;
        if ($referral_code) {
            $referrer = getUserByReferralCode($referral_code);
            if ($referrer) {
                $referrer_id = $referrer['id'];
            }
        }
        
        $user_id = createUser([
            'username' => $username,
            'phone' => $phone,
            'password' => password_hash($password, PASSWORD_DEFAULT),
            'referral_code' => generateReferralCode(),
            'referred_by' => $referrer_id,
            'balance' => getSetting('signup_bonus', 0)
        ]);
        
        if ($user_id) {
            // پاداش معرف
            if ($referrer_id) {
                $bonus = getSetting('referral_signup_bonus', 0);
                if ($bonus > 0) {
                    updateUserBalance($referrer_id, $bonus);
                    createTransaction($referrer_id, $bonus, 'bonus', 'پاداش معرفی کاربر جدید');
                }
            }
            
            $_SESSION['user_id'] = $user_id;
            jsonResponse(true, 'ثبت‌نام موفق', ['redirect' => '?page=dashboard']);
        } else {
            jsonResponse(false, 'خطا در ثبت‌نام');
        }
        break;
        
    case 'logout':
        session_destroy();
        jsonResponse(true, 'خروج موفق', ['redirect' => '?page=login']);
        break;
    
    // ============ داشبورد کاربر ============
    case 'user_dashboard':
        $configs = getUserActiveConfigs($user['id']);
        $devices = getUserDevices($user['id']);
        $referrals = getUserReferrals($user['id']);
        
        $daysLeft = 0;
        foreach ($configs as $config) {
            if ($config['days_left'] > $daysLeft) {
                $daysLeft = $config['days_left'];
            }
        }
        
        jsonResponse(true, '', [
            'active_configs' => count($configs),
            'connected_devices' => count($devices),
            'max_devices' => $user['max_devices'],
            'days_left' => $daysLeft,
            'referral_count' => count($referrals),
            'referral_reward' => getUserReferralReward($user['id']),
            'configs' => $configs
        ]);
        break;
    
    // ============ محصولات ============
    case 'get_products':
        $type = sanitize($_GET['type'] ?? 'all');
        $location = sanitize($_GET['location'] ?? '');
        $sort = sanitize($_GET['sort'] ?? 'popular');
        
        $products = getProducts($type, $location, $sort);
        jsonResponse(true, '', ['products' => $products]);
        break;
    
    // ============ تخفیف ============
    case 'check_discount':
        $code = sanitize($_GET['code'] ?? '');
        $product_id = (int)($_GET['product_id'] ?? 0);
        
        $discount = getDiscountByCode($code);
        
        if (!$discount) {
            jsonResponse(false, 'کد تخفیف معتبر نیست');
        }
        
        if ($discount['expires_at'] && strtotime($discount['expires_at']) < time()) {
            jsonResponse(false, 'کد تخفیف منقضی شده است');
        }
        
        if ($discount['max_uses'] && $discount['used_count'] >= $discount['max_uses']) {
            jsonResponse(false, 'ظرفیت استفاده از این کد تمام شده است');
        }
        
        jsonResponse(true, '', ['discount' => $discount['value']]);
        break;
    
    // ============ اطلاعات کارت بانکی ============
    case 'get_bank_card':
        jsonResponse(true, '', [
            'card_number' => getSetting('card_number', '6037-9974-1234-5678'),
            'card_holder' => getSetting('card_holder', 'نام صاحب کارت')
        ]);
        break;
    
    // ============ ثبت سفارش ============
    case 'create_order':
        $product_id = (int)($_POST['product_id'] ?? 0);
        $payment_method = sanitize($_POST['payment_method'] ?? 'wallet');
        $discount_code = sanitize($_POST['discount_code'] ?? '');
        $discount_percent = (int)($_POST['discount_percent'] ?? 0);
        
        $product = getProductById($product_id);
        if (!$product || $product['stock'] <= 0) {
            jsonResponse(false, 'محصول موجود نیست');
        }
        
        $price = $product['price'];
        $discount_amount = 0;
        
        if ($discount_percent > 0) {
            $discount_amount = floor($price * $discount_percent / 100);
            $price -= $discount_amount;
        }
        
        if ($payment_method === 'wallet') {
            if ($user['balance'] < $price) {
                jsonResponse(false, 'موجودی کیف پول کافی نیست');
            }
            
            // کسر از کیف پول
            updateUserBalance($user['id'], -$price);
            createTransaction($user['id'], -$price, 'purchase', 'خرید ' . $product['name']);
            
            // ثبت سفارش
            $order_id = createOrder([
                'user_id' => $user['id'],
                'product_id' => $product_id,
                'amount' => $price,
                'discount_amount' => $discount_amount,
                'payment_method' => 'wallet',
                'status' => 'completed'
            ]);
            
            // تخصیص کانفیگ
            assignConfigToUser($user['id'], $product_id, $order_id);
            
            // کمیسیون رفرال
            if ($user['referred_by']) {
                $commission = floor($price * getSetting('referral_commission', 10) / 100);
                if ($commission > 0) {
                    updateUserBalance($user['referred_by'], $commission);
                    createTransaction($user['referred_by'], $commission, 'referral', 'کمیسیون خرید زیرمجموعه');
                    createReferralCommission($user['referred_by'], $user['id'], $order_id, $price, $commission);
                }
            }
            
            jsonResponse(true, 'خرید با موفقیت انجام شد', ['payment_method' => 'wallet']);
            
        } else {
            // کارت به کارت
            $receipt = null;
            if (isset($_FILES['receipt']) && $_FILES['receipt']['error'] === UPLOAD_ERR_OK) {
                $receipt = uploadFile($_FILES['receipt'], 'receipts');
            }
            
            $order_id = createOrder([
                'user_id' => $user['id'],
                'product_id' => $product_id,
                'amount' => $price,
                'discount_amount' => $discount_amount,
                'payment_method' => 'card',
                'receipt_image' => $receipt,
                'status' => 'pending'
            ]);
            
            jsonResponse(true, 'سفارش ثبت شد', ['payment_method' => 'card', 'order_id' => $order_id]);
        }
        break;
    
    // ============ شارژ کیف پول ============
    case 'request_charge':
        $amount = (int)($_POST['amount'] ?? 0);
        
        if ($amount < getSetting('min_wallet_charge', 10000)) {
            jsonResponse(false, 'حداقل مبلغ شارژ ' . number_format(getSetting('min_wallet_charge', 10000)) . ' تومان است');
        }
        
        $receipt = null;
        if (isset($_FILES['receipt']) && $_FILES['receipt']['error'] === UPLOAD_ERR_OK) {
            $receipt = uploadFile($_FILES['receipt'], 'receipts');
        }
        
        if (!$receipt) {
            jsonResponse(false, 'لطفا تصویر رسید را آپلود کنید');
        }
        
        createChargeRequest([
            'user_id' => $user['id'],
            'amount' => $amount,
            'receipt_image' => $receipt,
            'status' => 'pending'
        ]);
        
        jsonResponse(true, 'درخواست شارژ ثبت شد');
        break;
    
    // ============ تراکنش‌ها ============
    case 'get_transactions':
        $type = sanitize($_GET['type'] ?? '');
        $page = (int)($_GET['page'] ?? 1);
        
        $transactions = getUserTransactions($user['id'], $type, $page);
        jsonResponse(true, '', ['transactions' => $transactions]);
        break;
    
    // ============ دریافت کانفیگ ============
    case 'get_config_data':
        $config_id = (int)($_GET['id'] ?? 0);
        
        $config = getUserConfig($user['id'], $config_id);
        if (!$config) {
            jsonResponse(false, 'کانفیگ یافت نشد');
        }
        
        jsonResponse(true, '', ['config_data' => $config['config_data']]);
        break;
    
    // ============ تیکت‌ها ============
    case 'user_get_tickets':
        $tickets = getUserTickets($user['id']);
        jsonResponse(true, '', ['tickets' => $tickets]);
        break;
        
    case 'user_get_ticket':
        $ticket_id = (int)($_GET['id'] ?? 0);
        $ticket = getTicketWithMessages($ticket_id, $user['id']);
        
        if (!$ticket) {
            jsonResponse(false, 'تیکت یافت نشد');
        }
        
        jsonResponse(true, '', ['ticket' => $ticket]);
        break;
        
    case 'create_ticket':
        $subject = sanitize($_POST['subject'] ?? '');
        $message = sanitize($_POST['message'] ?? '');
        $department = sanitize($_POST['department'] ?? 'general');
        $priority = sanitize($_POST['priority'] ?? 'medium');
        
        if (empty($subject) || empty($message)) {
            jsonResponse(false, 'لطفا تمام فیلدها را پر کنید');
        }
        
        $attachment = null;
        if (isset($_FILES['attachment']) && $_FILES['attachment']['error'] === UPLOAD_ERR_OK) {
            $attachment = uploadFile($_FILES['attachment'], 'tickets');
        }
        
        $ticket_id = createTicket([
            'user_id' => $user['id'],
            'subject' => $subject,
            'department' => $department,
            'priority' => $priority,
            'status' => 'open'
        ]);
        
        createTicketMessage([
            'ticket_id' => $ticket_id,
            'user_id' => $user['id'],
            'message' => $message,
            'attachment' => $attachment,
            'is_admin' => false
        ]);
        
        jsonResponse(true, 'تیکت ارسال شد');
        break;
        
    case 'reply_ticket':
        $ticket_id = (int)($_POST['ticket_id'] ?? 0);
        $message = sanitize($_POST['message'] ?? '');
        
        if (empty($message)) {
            jsonResponse(false, 'لطفا پیام را وارد کنید');
        }
        
        $ticket = getTicketById($ticket_id);
        if (!$ticket || $ticket['user_id'] != $user['id']) {
            jsonResponse(false, 'تیکت یافت نشد');
        }
        
        if ($ticket['status'] === 'closed') {
            jsonResponse(false, 'این تیکت بسته شده است');
        }
        
        $attachment = null;
        if (isset($_FILES['attachment']) && $_FILES['attachment']['error'] === UPLOAD_ERR_OK) {
            $attachment = uploadFile($_FILES['attachment'], 'tickets');
        }
        
        createTicketMessage([
            'ticket_id' => $ticket_id,
            'user_id' => $user['id'],
            'message' => $message,
            'attachment' => $attachment,
            'is_admin' => false
        ]);
        
        updateTicketStatus($ticket_id, 'waiting');
        
        jsonResponse(true, 'پاسخ ارسال شد');
        break;
    
    // ============ API های ادمین ============
    // ... (API های ادمین مشابه قبلی)
    
    default:
        jsonResponse(false, 'درخواست نامعتبر');
}

// ============ توابع کمکی ============

function jsonResponse($success, $message = '', $data = []) {
    echo json_encode(array_merge(['success' => $success, 'message' => $message], $data), JSON_UNESCAPED_UNICODE);
    exit;
}

function sanitize($input) {
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

function generateReferralCode($length = 8) {
    return strtoupper(substr(md5(uniqid(mt_rand(), true)), 0, $length));
}

function uploadFile($file, $folder) {
    $uploadDir = 'uploads/' . $folder . '/';
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
    $filename = uniqid() . '.' . $ext;
    $destination = $uploadDir . $filename;
    
    if (move_uploaded_file($file['tmp_name'], $destination)) {
        return $destination;
    }
    
    return null;
}
?>