<?php
// includes/Database.php

class Database {
    private static $instance = null;
    private $pdo;
    
    private function __construct() {
        try {
            $this->pdo = new PDO(
                "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4",
                DB_USER,
                DB_PASS,
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false
                ]
            );
        } catch (PDOException $e) {
            throw new Exception("Database connection failed: " . $e->getMessage());
        }
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function getConnection() {
        return $this->pdo;
    }
    
    // ذخیره کانال
    public function saveChannel($username, $title = null) {
        $stmt = $this->pdo->prepare("
            INSERT INTO channels (username, title, last_scan) 
            VALUES (?, ?, NOW())
            ON DUPLICATE KEY UPDATE title = VALUES(title), last_scan = NOW()
        ");
        $stmt->execute([$username, $title]);
        return $this->pdo->lastInsertId() ?: $this->getChannelId($username);
    }
    
    public function getChannelId($username) {
        $stmt = $this->pdo->prepare("SELECT id FROM channels WHERE username = ?");
        $stmt->execute([$username]);
        $result = $stmt->fetch();
        return $result ? $result['id'] : null;
    }
    
    // ذخیره کانفیگ
    public function saveConfig($channelId, $data) {
        $hash = hash('sha256', $data['config_raw']);
        
        // چک کردن تکراری
        $stmt = $this->pdo->prepare("SELECT id, found_count, score FROM configs WHERE config_hash = ?");
        $stmt->execute([$hash]);
        $existing = $stmt->fetch();
        
        if ($existing) {
            // افزایش امتیاز اگر از کانال متفاوت
            $stmt = $this->pdo->prepare("
                UPDATE configs SET 
                    found_count = found_count + 1,
                    score = score + 1,
                    last_seen = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$existing['id']]);
            return $existing['id'];
        }
        
        $stmt = $this->pdo->prepare("
            INSERT INTO configs 
            (channel_id, protocol, config_raw, config_hash, server, port, remark, is_tls, is_reality, transport, last_seen)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $channelId,
            $data['protocol'],
            $data['config_raw'],
            $hash,
            $data['server'] ?? null,
            $data['port'] ?? null,
            $data['remark'] ?? null,
            $data['is_tls'] ?? 0,
            $data['is_reality'] ?? 0,
            $data['transport'] ?? null
        ]);
        
        return $this->pdo->lastInsertId();
    }
    
    // دریافت کانفیگ‌ها
    public function getConfigs($filters = []) {
        $where = ["c.score >= " . MIN_SCORE_DISPLAY];
        $params = [];
        
        if (!empty($filters['protocol'])) {
            $where[] = "c.protocol = ?";
            $params[] = $filters['protocol'];
        }
        
        if (!empty($filters['light_mode'])) {
            $where[] = "c.is_tls = 0 AND c.is_reality = 0";
        }
        
        if (!empty($filters['channel'])) {
            $where[] = "ch.username = ?";
            $params[] = $filters['channel'];
        }
        
        $whereClause = implode(' AND ', $where);
        $limit = $filters['limit'] ?? 50;
        $offset = $filters['offset'] ?? 0;
        
        $stmt = $this->pdo->prepare("
            SELECT c.*, ch.username as channel_name, ch.title as channel_title
            FROM configs c
            LEFT JOIN channels ch ON c.channel_id = ch.id
            WHERE $whereClause
            ORDER BY c.score DESC, c.last_seen DESC
            LIMIT ? OFFSET ?
        ");
        
        $params[] = (int)$limit;
        $params[] = (int)$offset;
        
        $stmt->execute($params);
        return $stmt->fetchAll();
    }
    
    // آمار
    public function getStats() {
        $stats = [];
        
        $stmt = $this->pdo->query("SELECT COUNT(*) as total FROM configs");
        $stats['total_configs'] = $stmt->fetch()['total'];
        
        $stmt = $this->pdo->query("SELECT COUNT(*) as total FROM channels WHERE is_active = 1");
        $stats['total_channels'] = $stmt->fetch()['total'];
        
        $stmt = $this->pdo->query("
            SELECT protocol, COUNT(*) as count 
            FROM configs 
            GROUP BY protocol
        ");
        $stats['by_protocol'] = $stmt->fetchAll();
        
        return $stats;
    }
    
    public function updateStats() {
        $this->pdo->exec("
            UPDATE stats SET 
                total_scans = total_scans + 1,
                total_configs = (SELECT COUNT(*) FROM configs),
                last_update = NOW()
        ");
    }
}