<?php
// includes/ConfigParser.php

class ConfigParser {
    private $debug = [];
    
    // استخراج همه کانفیگ‌ها از متن
    public function extractConfigs($text, $channelName = '') {
        $configs = [];
        
        // الگوهای regex بهتر
        $patterns = [
            'vless' => '/vless:\/\/[a-f0-9\-]{36}@[^\s<>"\'#]+(?:#[^\s<>"\']*)?/i',
            'vmess' => '/vmess:\/\/[A-Za-z0-9+\/=]{50,}/i',
            'trojan' => '/trojan:\/\/[^\s<>"\'@]+@[^\s<>"\'#]+(?:#[^\s<>"\']*)?/i',
            'ss' => '/ss:\/\/[A-Za-z0-9+\/=@:%\-_.]+(?:#[^\s<>"\']*)?/i',
        ];
        
        foreach ($patterns as $protocol => $pattern) {
            preg_match_all($pattern, $text, $matches);
            
            $this->debug[] = "$protocol found: " . count($matches[0]);
            
            foreach ($matches[0] as $configRaw) {
                // پاکسازی
                $configRaw = trim($configRaw);
                $configRaw = preg_replace('/[<>"\'].*$/', '', $configRaw);
                
                // اعتبارسنجی اولیه
                if (strlen($configRaw) < 30) continue;
                
                $parsed = $this->parseConfig($configRaw, $protocol, $channelName);
                if ($parsed) {
                    $configs[] = $parsed;
                }
            }
        }
        
        $this->debug[] = "Total configs extracted: " . count($configs);
        
        return $configs;
    }
    
    // پارس کردن یک کانفیگ
    public function parseConfig($configRaw, $protocol, $channelName = '') {
        $data = [
            'protocol' => $protocol,
            'config_raw' => $configRaw,
            'server' => null,
            'port' => null,
            'remark' => null,
            'is_tls' => 0,
            'is_reality' => 0,
            'transport' => 'tcp'
        ];
        
        try {
            switch ($protocol) {
                case 'vless':
                    $data = array_merge($data, $this->parseVless($configRaw));
                    break;
                case 'vmess':
                    $data = array_merge($data, $this->parseVmess($configRaw));
                    break;
                case 'trojan':
                    $data = array_merge($data, $this->parseTrojan($configRaw));
                    break;
                case 'ss':
                    $data = array_merge($data, $this->parseShadowsocks($configRaw));
                    break;
            }
            
            // تغییر نام به اسم کانال
            if ($channelName) {
                $data['config_raw'] = $this->renameConfig($configRaw, $protocol, $channelName);
                $data['remark'] = $channelName;
            }
            
        } catch (Exception $e) {
            $this->debug[] = "Parse error: " . $e->getMessage();
            return null;
        }
        
        return $data;
    }
    
    // پارس VLESS
    private function parseVless($config) {
        $result = [];
        
        // vless://uuid@server:port?params#remark
        if (!preg_match('/vless:\/\/([^@]+)@([^:]+):(\d+)/', $config, $matches)) {
            return $result;
        }
        
        $result['server'] = $matches[2];
        $result['port'] = (int)$matches[3];
        
        // استخراج remark
        if (preg_match('/#(.+)$/', $config, $remarkMatch)) {
            $result['remark'] = urldecode($remarkMatch[1]);
        }
        
        // استخراج پارامترها
        if (preg_match('/\?([^#]+)/', $config, $paramMatch)) {
            parse_str($paramMatch[1], $params);
            
            $security = $params['security'] ?? '';
            $result['is_tls'] = ($security === 'tls') ? 1 : 0;
            $result['is_reality'] = ($security === 'reality') ? 1 : 0;
            $result['transport'] = $params['type'] ?? 'tcp';
        }
        
        return $result;
    }
    
    // پارس VMESS
    private function parseVmess($config) {
        $result = [];
        
        $base64 = str_replace('vmess://', '', $config);
        
        // پد کردن base64
        $base64 = str_pad($base64, strlen($base64) + (4 - strlen($base64) % 4) % 4, '=');
        
        $json = @base64_decode($base64);
        if (!$json) return $result;
        
        $data = @json_decode($json, true);
        if (!$data) return $result;
        
        $result['server'] = $data['add'] ?? null;
        $result['port'] = isset($data['port']) ? (int)$data['port'] : null;
        $result['remark'] = $data['ps'] ?? null;
        $result['is_tls'] = ($data['tls'] ?? '') === 'tls' ? 1 : 0;
        $result['transport'] = $data['net'] ?? 'tcp';
        
        return $result;
    }
    
    // پارس Trojan
    private function parseTrojan($config) {
        $result = [];
        
        if (!preg_match('/trojan:\/\/([^@]+)@([^:]+):(\d+)/', $config, $matches)) {
            return $result;
        }
        
        $result['server'] = $matches[2];
        $result['port'] = (int)$matches[3];
        $result['is_tls'] = 1; // Trojan همیشه TLS
        
        if (preg_match('/#(.+)$/', $config, $remarkMatch)) {
            $result['remark'] = urldecode($remarkMatch[1]);
        }
        
        return $result;
    }
    
    // پارس Shadowsocks
    private function parseShadowsocks($config) {
        $result = [];
        $result['protocol'] = 'ss';
        
        $config = preg_replace('/^ss:\/\//', '', $config);
        
        // استخراج remark
        if (preg_match('/#(.+)$/', $config, $remarkMatch)) {
            $result['remark'] = urldecode($remarkMatch[1]);
            $config = preg_replace('/#.+$/', '', $config);
        }
        
        // فرمت جدید: base64@server:port
        if (preg_match('/^([^@]+)@([^:]+):(\d+)/', $config, $matches)) {
            $result['server'] = $matches[2];
            $result['port'] = (int)$matches[3];
        }
        // فرمت قدیمی: همه چیز base64
        else {
            $decoded = @base64_decode($config);
            if ($decoded && preg_match('/@([^:]+):(\d+)/', $decoded, $matches)) {
                $result['server'] = $matches[1];
                $result['port'] = (int)$matches[2];
            }
        }
        
        return $result;
    }
    
    // تغییر نام کانفیگ
    public function renameConfig($configRaw, $protocol, $newName) {
        $newNameEncoded = urlencode($newName);
        
        switch ($protocol) {
            case 'vless':
            case 'trojan':
            case 'ss':
                // حذف نام قبلی
                $config = preg_replace('/#[^#]*$/', '', $configRaw);
                return $config . '#' . $newNameEncoded;
                
            case 'vmess':
                $base64 = str_replace('vmess://', '', $configRaw);
                $base64 = str_pad($base64, strlen($base64) + (4 - strlen($base64) % 4) % 4, '=');
                $json = @base64_decode($base64);
                $data = @json_decode($json, true);
                
                if ($data) {
                    $data['ps'] = $newName;
                    return 'vmess://' . base64_encode(json_encode($data, JSON_UNESCAPED_UNICODE));
                }
                return $configRaw;
        }
        
        return $configRaw;
    }
    
    public function getDebug() {
        return $this->debug;
    }
}