// assets/app.js

// State
let currentFilter = 'all';
let configs = [];

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    loadConfigs();
    initFilters();
});
async function startScan() {
    const input = document.getElementById('channelInput');
    const postCount = document.getElementById('postCount').value;
    const btn = document.getElementById('scanBtn');
    const results = document.getElementById('results');
    const resultsList = document.getElementById('resultsList');
    
    const channels = input.value.trim().split('\n').filter(c => c.trim());
    
    if (channels.length === 0) {
        alert('لطفاً حداقل یک کانال وارد کنید');
        return;
    }
    
    // Disable button
    btn.disabled = true;
    btn.innerHTML = `
        <svg class="w-5 h-5 animate-spin" viewBox="0 0 24 24">
            <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4" fill="none"/>
            <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4z"/>
        </svg>
        در حال اسکن...
    `;
    
    try {
        const response = await fetch('api/scan.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ 
                channels,
                post_count: parseInt(postCount),
                debug: false
            })
        });
        
        const data = await response.json();
        
        results.classList.remove('hidden');
        resultsList.innerHTML = '';
        
        if (!data.success) {
            resultsList.innerHTML = `
                <div class="glass-card rounded-xl p-4 text-center text-red-400">
                    ❌ ${data.error || 'خطا در اسکن'}
                </div>
            `;
            return;
        }
        
        // Summary
        resultsList.innerHTML = `
            <div class="glass-card rounded-xl p-4 mb-4">
                <div class="grid grid-cols-2 gap-4 text-center">
                    <div>
                        <div class="text-2xl font-bold text-indigo-400">${data.channels_scanned}</div>
                        <div class="text-xs text-gray-400">کانال اسکن شده</div>
                    </div>
                    <div>
                        <div class="text-2xl font-bold text-green-400">${data.configs_found}</div>
                        <div class="text-xs text-gray-400">کانفیگ یافت شده</div>
                    </div>
                </div>
            </div>
        `;
        
        // Details
        data.details.forEach(detail => {
            const isSuccess = detail.status === 'success';
            const statusClass = isSuccess ? 'border-green-500/30' : 'border-red-500/30';
            const statusIcon = isSuccess ? '✅' : '❌';
            
            resultsList.innerHTML += `
                <div class="glass-card rounded-xl p-4 ${statusClass} border">
                    <div class="flex items-center justify-between">
                        <div>
                            <div class="font-medium">${statusIcon} @${detail.channel}</div>
                            <div class="text-xs text-gray-400">${detail.title || ''}</div>
                        </div>
                        <div class="text-left">
                            ${isSuccess 
                                ? `<span class="text-green-400 font-bold">${detail.configs}</span> <span class="text-xs text-gray-400">کانفیگ</span>` 
                                : `<span class="text-xs text-red-400">${detail.message}</span>`
                            }
                        </div>
                    </div>
                </div>
            `;
        });
        
    } catch (error) {
        resultsList.innerHTML = `
            <div class="glass-card rounded-xl p-4 text-center text-red-400">
                ❌ خطا: ${error.message}
            </div>
        `;
    } finally {
        btn.disabled = false;
        btn.innerHTML = `
            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
            </svg>
            شروع اسکن
        `;
    }
}
// Load Configs
async function loadConfigs() {
    const container = document.getElementById('configList');
    
    try {
        let url = 'api/configs.php?limit=50';
        
        if (currentFilter && currentFilter !== 'all') {
            if (currentFilter === 'light') {
                url += '&light_mode=1';
            } else {
                url += `&protocol=${currentFilter}`;
            }
        }
        
        const response = await fetch(url);
        const data = await response.json();
        
        if (data.success) {
            configs = data.configs;
            updateStats(data.stats);
            renderConfigs();
        }
        
    } catch (error) {
        container.innerHTML = `
            <div class="glass-card rounded-xl p-6 text-center">
                <div class="text-red-400 text-4xl mb-2">⚠️</div>
                <p class="text-gray-400">خطا در بارگذاری</p>
                <button onclick="loadConfigs()" class="mt-3 px-4 py-2 rounded-lg bg-white/10 hover:bg-white/20 transition">
                    تلاش مجدد
                </button>
            </div>
        `;
    }
}

// Update Stats
function updateStats(stats) {
    document.getElementById('statConfigs').textContent = formatNumber(stats.total_configs);
    document.getElementById('statChannels').textContent = formatNumber(stats.total_channels);
    document.getElementById('statOnline').textContent = '✓';
}

// Render Configs
function renderConfigs() {
    const container = document.getElementById('configList');
    
    if (configs.length === 0) {
        container.innerHTML = `
            <div class="glass-card rounded-xl p-6 text-center">
                <div class="text-4xl mb-2">📭</div>
                <p class="text-gray-400">کانفیگی یافت نشد</p>
                <a href="scanner.php" class="inline-block mt-3 px-4 py-2 rounded-lg bg-indigo-500 hover:bg-indigo-600 transition">
                    اسکن کانال
                </a>
            </div>
        `;
        return;
    }
    
    container.innerHTML = configs.map(config => createConfigCard(config)).join('');
}

// Create Config Card
function createConfigCard(config) {
    const protocolBadge = getProtocolBadge(config.protocol);
    const scoreBadge = getScoreBadge(config.score);
    const shortConfig = config.config_raw.substring(0, 80) + '...';
    
    return `
        <div class="glass-card rounded-xl p-4 space-y-3">
            <!-- Header -->
            <div class="flex items-center justify-between">
                <div class="flex items-center gap-2">
                    <span class="${protocolBadge} px-2 py-1 rounded text-xs font-bold uppercase">
                        ${config.protocol}
                    </span>
                    ${scoreBadge}
                </div>
                <span class="text-xs text-gray-400">
                    @${config.channel_name || 'unknown'}
                </span>
            </div>
            
            <!-- Config Preview -->
            <div class="bg-black/30 rounded-lg p-3">
                <code class="config-text text-gray-300">${escapeHtml(shortConfig)}</code>
            </div>
            
            <!-- Server Info -->
            <div class="flex items-center gap-4 text-xs text-gray-400">
                <span>🌐 ${config.server || 'N/A'}</span>
                <span>🚪 ${config.port || 'N/A'}</span>
                ${config.is_tls ? '<span class="text-green-400">🔒 TLS</span>' : ''}
                ${config.is_reality ? '<span class="text-purple-400">✨ Reality</span>' : ''}
            </div>
            
            <!-- Actions -->
            <div class="flex gap-2">
                <button 
                    onclick='copyConfig(${JSON.stringify(config.config_raw)})'
                    class="flex-1 py-2 rounded-lg bg-indigo-500/20 hover:bg-indigo-500/30 text-indigo-400 transition flex items-center justify-center gap-2"
                >
                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 5H6a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2v-1M8 5a2 2 0 002 2h2a2 2 0 002-2M8 5a2 2 0 012-2h2a2 2 0 012 2m0 0h2a2 2 0 012 2v3m2 4H10m0 0l3-3m-3 3l3 3"/>
                    </svg>
                    کپی
                </button>
                <button 
                    onclick='showQR(${JSON.stringify(config.config_raw)})'
                    class="px-4 py-2 rounded-lg bg-white/5 hover:bg-white/10 transition"
                >
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v1m6 11h2m-6 0h-2v4m0-11v3m0 0h.01M12 12h4.01M16 20h4M4 12h4m12 0h.01M5 8h2a1 1 0 001-1V5a1 1 0 00-1-1H5a1 1 0 00-1 1v2a1 1 0 001 1zm12 0h2a1 1 0 001-1V5a1 1 0 00-1-1h-2a1 1 0 00-1 1v2a1 1 0 001 1zM5 20h2a1 1 0 001-1v-2a1 1 0 00-1-1H5a1 1 0 00-1 1v2a1 1 0 001 1z"/>
                    </svg>
                </button>
            </div>
        </div>
    `;
}

// Copy Config
function copyConfig(config) {
    navigator.clipboard.writeText(config).then(() => {
        showToast('✓ کانفیگ کپی شد');
    }).catch(() => {
        // Fallback
        const textarea = document.createElement('textarea');
        textarea.value = config;
        document.body.appendChild(textarea);
        textarea.select();
        document.execCommand('copy');
        document.body.removeChild(textarea);
        showToast('✓ کانفیگ کپی شد');
    });
}

// Show QR
function showQR(config) {
    const modal = document.getElementById('qrModal');
    const img = document.getElementById('qrImage');
    img.src = `api/qr.php?config=${encodeURIComponent(config)}`;
    modal.classList.remove('hidden');
    modal.classList.add('flex');
}

// Close QR
function closeQR(event) {
    const modal = document.getElementById('qrModal');
    if (!event || event.target === modal) {
        modal.classList.add('hidden');
        modal.classList.remove('flex');
    }
}

// Show Toast
function showToast(message) {
    const toast = document.getElementById('toast');
    const toastMessage = document.getElementById('toastMessage');
    toastMessage.textContent = message;
    toast.classList.add('show');
    
    setTimeout(() => {
        toast.classList.remove('show');
    }, 2000);
}

// Init Filters
function initFilters() {
    document.querySelectorAll('.filter-pill').forEach(pill => {
        pill.addEventListener('click', () => {
            document.querySelectorAll('.filter-pill').forEach(p => p.classList.remove('active'));
            pill.classList.add('active');
            currentFilter = pill.dataset.filter;
            loadConfigs();
        });
    });
}

// Refresh
document.getElementById('refreshBtn')?.addEventListener('click', () => {
    loadConfigs();
    showToast('🔄 بروزرسانی شد');
});

// Helpers
function getProtocolBadge(protocol) {
    const badges = {
        'vless': 'badge-vless',
        'vmess': 'badge-vmess',
        'trojan': 'badge-trojan',
        'ss': 'badge-ss'
    };
    return badges[protocol] || 'bg-gray-500';
}

function getScoreBadge(score) {
    if (score >= 5) {
        return '<span class="score-high px-2 py-0.5 rounded text-xs">⭐ ' + score + '</span>';
    } else if (score >= 3) {
        return '<span class="score-medium px-2 py-0.5 rounded text-xs">🔥 ' + score + '</span>';
    }
    return '';
}

function formatNumber(num) {
    if (num >= 1000) {
        return (num / 1000).toFixed(1) + 'K';
    }
    return num;
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Subscription Modal
function showSubscription() {
    const url = 'api/subscription.php';
    
    const html = `
        <div class="fixed inset-0 z-[100] flex items-center justify-center bg-black/80" onclick="this.remove()">
            <div class="glass-card rounded-2xl p-6 max-w-sm mx-4" onclick="event.stopPropagation()">
                <h3 class="text-lg font-bold mb-4 text-center">📥 لینک اشتراک</h3>
                
                <div class="space-y-3">
                    <a href="${url}?format=base64" target="_blank" class="block p-3 rounded-lg bg-indigo-500/20 hover:bg-indigo-500/30 transition">
                        <div class="font-medium">Base64 (پیش‌فرض)</div>
                        <div class="text-xs text-gray-400">برای اکثر کلاینت‌ها</div>
                    </a>
                    
                    <a href="${url}?format=raw" target="_blank" class="block p-3 rounded-lg bg-purple-500/20 hover:bg-purple-500/30 transition">
                        <div class="font-medium">Raw Text</div>
                        <div class="text-xs text-gray-400">متن خام</div>
                    </a>
                    
                    <a href="${url}?light=1" target="_blank" class="block p-3 rounded-lg bg-green-500/20 hover:bg-green-500/30 transition">
                        <div class="font-medium">⚡ حالت سبک</div>
                        <div class="text-xs text-gray-400">بدون TLS و Reality</div>
                    </a>
                </div>
                
                <button onclick="this.closest('.fixed').remove()" class="w-full mt-4 py-2 rounded-lg bg-white/10 hover:bg-white/20 transition">
                    بستن
                </button>
            </div>
        </div>
    `;
    
    document.body.insertAdjacentHTML('beforeend', html);
}