<?php
// api/scan.php

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once '../config.php';
require_once '../includes/Database.php';
require_once '../includes/TelegramScraper.php';
require_once '../includes/ConfigParser.php';

try {
    // دریافت ورودی
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        // شاید از GET اومده
        $input = $_GET;
    }
    
    $channels = $input['channels'] ?? [];
    $postCount = (int)($input['post_count'] ?? 50);
    $debugMode = isset($input['debug']) && $input['debug'];
    
    // اگر channels رشته بود
    if (is_string($channels)) {
        $channels = array_filter(explode("\n", $channels));
    }
    
    if (empty($channels)) {
        throw new Exception('لطفاً حداقل یک کانال وارد کنید');
    }
    
    // محدودیت
    $channels = array_slice($channels, 0, 10);
    $postCount = min(max($postCount, 10), 100);
    
    $db = Database::getInstance();
    $scraper = new TelegramScraper();
    $parser = new ConfigParser();
    
    $results = [
        'success' => true,
        'channels_scanned' => 0,
        'configs_found' => 0,
        'configs_saved' => 0,
        'details' => [],
        'debug' => []
    ];
    
    foreach ($channels as $channelInput) {
        $channelInput = trim($channelInput);
        if (empty($channelInput)) continue;
        
        $channelResult = [
            'channel' => $channelInput,
            'title' => null,
            'configs' => 0,
            'status' => 'pending',
            'message' => '',
            'debug' => []
        ];
        
        try {
            // اسکرپ کانال
            $channelData = $scraper->scrapeChannel($channelInput, $postCount);
            $channelName = $channelData['username'];
            $channelResult['title'] = $channelData['title'];
            $channelResult['debug'] = $channelData['debug'] ?? [];
            
            // ذخیره کانال
            $channelId = $db->saveChannel($channelName, $channelData['title']);
            
            $channelConfigs = 0;
            
            // استخراج کانفیگ از هر پیام
            foreach ($channelData['messages'] as $message) {
                $configs = $parser->extractConfigs($message, $channelName);
                
                foreach ($configs as $config) {
                    try {
                        $db->saveConfig($channelId, $config);
                        $channelConfigs++;
                    } catch (Exception $e) {
                        // تکراری یا خطای دیگر
                    }
                }
            }
            
            $channelResult['configs'] = $channelConfigs;
            $channelResult['status'] = 'success';
            $channelResult['message'] = "$channelConfigs کانفیگ یافت شد";
            $channelResult['debug'] = array_merge(
                $channelResult['debug'], 
                $parser->getDebug()
            );
            
            $results['channels_scanned']++;
            $results['configs_found'] += $channelConfigs;
            
        } catch (Exception $e) {
            $channelResult['status'] = 'error';
            $channelResult['message'] = $e->getMessage();
        }
        
        $results['details'][] = $channelResult;
        
        // تاخیر بین درخواست‌ها
        usleep(500000);
    }
    
    // بروزرسانی آمار
    $db->updateStats();
    
    // حذف دیباگ در حالت عادی
    if (!$debugMode) {
        unset($results['debug']);
        foreach ($results['details'] as &$detail) {
            unset($detail['debug']);
        }
    }
    
    echo json_encode($results, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}